Attribute VB_Name = "modRC3"
Option Explicit

'*******************************************************************************
'* modRC3
'*
'*      (c) 2009 Ulrich Korndrfer proSource software development
'*          www.prosource.de
'*          German site with VB articles (in english) and code (comments in english)
'*
'* * Introduction
'*
'* This module is a helper module for the RichClient3 framework of Olaf Schmidt
'* (www.datenhaus.de), available from www.datenhaus.de/Downloads/dhRichClient3.zip,
'* and is an alternative to Olaf's modFactory.bas.
'*
'* The three framework dlls (see "Module dependencies" below) may reside at any
'* place you wish, as long as all three dlls are in the same folder. So the
'* prerequisite to use this module is:
'*
'* (0) Copy the three framework dlls to the same folder. The folder may be at any
'*     place, but this three files must be in this folder. I will refer to this
'*     folder as the framework folder.
'*     This must be done on the developer's machine and also on the customer's
'*     machine (actual folder pathes then of course may differ).
'*
'* The module loads the framework dll dhRichCLient3.dll, instantiates the cFactory,
'* cConstructor and cRegFree classes and gives access to these using gRC3_Factory,
'* gRC3_New and gRC3_RegFree.
'*
'* It can instantiate those instances using the registry. Then dhRichClient3.dll must
'* be registered.

'* It can also instantiate without using the registry (regfree). Then gRC3_Init must
'* be called with parameter FrameworkPath pointing to the framework folder.
'* dhRichClient3.dll then may be registered or not, it doesn't matter.
'*
'* When running the application in the IDE, the module uses the registered version
'* of dhRichClient3.dll if param ForceRegFree of gRC3_Init is False, otherwise it
'* uses the copy of dhRichClient3.dll from the framework folder in a "regfree" way.
'*
'* When running compiled outside the IDE, the module always uses the copy of
'* dhRichClient3.dll from the framework folder in a "regfree" way, no matter what
'* value param ForceRegFree has.
'*
'* Instantiating regfree allows to test eg. the regfree performance of applications
'* already on the developer's machine.
'*
'* The same module (unaltered) can be used with compiled and deployed applications.
'* Just copy the framework dlls to the customer machine, without the need to register
'* the dhRichClient3.dll on the customer's machine.
'*
'* * Module usage on developer machines, when developing your application
'*
'* (1) Execute step (0) from above !!!
'* (2) Register the dhRichClient3.dll !!! This is mandatory (on the developer machine),
'*     because the module works "early bound" and therefore needs the dll registered.
'*     This is a prerequisite for running in the IDE and for being able to compile
'*     the application. It is not more necessary when the application is compiled
'*     and runs compiled. As a benefit you get Intellisense when developing and performance
'*     is better when running (in and outside the IDE).
'* (2) Include this module in your projects.
'* (3) Call gRC3_Init. The call must be made at least just before anything of the
'*     framework is used. A good place for not to forget the call would be in a
'*     Sub Main or in the Form_Initialize or Form_Load events. gRC3_Init can be called
'*     as often you want, as it returns immediately without doing anything else after
'*     it has been called once.
'*     If you decide to work regfree when running in the IDE, set the ForceRegFree param
'*     to True. In any case the FrameworkPath param should point to the framework folder.
'* (4) Use gRC3_Factory, gRC3_New and gRC3_RegFree to get access to instances
'*     of the framework's cFactory, cConstructor and cRegFree classes.
'* (5) No shutdown code is necessary. Simply exit the application.
'*
'* * Module usage on a customer machine, to which you have deployed your compiled application
'*
'* (1) Execute step (0) from above!
'* (2) Somehow tell the application the path to the framework folder. The application
'*     then has to call gRC3_Init with the FrameworkPath param pointing to this path.
'*     Registering of dhRichClient3.dll is *not* necessary. However it *may* be
'*     registered, the module will work anyway.
'*     As a summary: - param FrameworkPath of gRC3_Init must point to the framework folder
'*                   - the value of param ForceRegFree does not matter (always work reg free)
'*
'* * Strategies for choosing a framework folder path
'*
'* - Set the framework folder path to the application directory
'*
'*   Pros: - uses same (relative) path for development and deployed application, no
'*           need to differentiate between running compiled on the developer or
'*           customer machine.
'*   Cons: - if you want to test the application by running compiled on the
'*           development machine you must copy the framework dlls in every application
'*           folder.
'*         - your application directory gets cluttered with dlls.
'*
'* - Set the framework folder path to a subfolder of the application folder
'*
'*   Pros: - same as above
'*         - no application directory cluttering
'*   Cons: - same as above (except cluttering)
'*
'* - Set the framework folder path on the developer machine to a "central" folder,
'*   on the customer machine use the application folder's subfolder for each
'*   of your applications or again a "central" folder your applications share.
'*
'*   Pros: - runs well on developer machine, all your applications you are developing
'*           share the same framework folder, no need to further copy framework dlls
'*           (there is exactly one folder on the developer machine used by all apps).
'*           Same holds for the customer machine if also a "central" folder is used.
'*         - no application folder dll cluttering
'*   Cons: - you have to switch the framework folder pathes dependend on wether you
'*           run on the developer or the customer machine.
'*
'* * Module dependencies: depends on the 3 RichClient3 framework dlls, that currently are:
'*
'*                        - DirectCOM.dll (the version with GetInstanceEx function)
'*                        - dhRichClient3.dll
'*                        - sqlite36_engine.dll
'*
'*                        and some apis (because of UNICODE usage WIN versions should be
'*                        NT4, 2K or above)
'*
'* Version 1.3  from 2009-03-11
'* Version 1.2  from 2009-03-07
'* Version 1.1  from 2009-03-06
'* Version 1.0  from 2009-03-03
'*
'*******************************************************************************


'*******************************************************************************
'* Private API declares and consts
'*******************************************************************************

'GetInstanceEx instantiates a class named ClassName in an Active-X dll (aka COM-Server)
'located at FilePath. It is the work horse that makes regfree COM possible, because
'it does not matter wether the ActiveX dll is registered or not! In this module it
'is called exactly once in gRC3_Init, if dhRichClient3 shall be used reg free.

Private Declare Function GetInstanceEx Lib "DirectCOM.dll" _
                  (ByRef StrPtr_FilePath As Long, _
                   ByRef StrPtr_ClassName As Long, _
          Optional ByVal UseAlteredSearchPath As Boolean = True) As Object

'Returns an error message when a call to GetInstanceEx failed

Private Declare Function GETINSTANCELASTERROR Lib "DirectCOM.dll" _
                         () As String

'*****
'* OS helper apis. As they are using the UNICODE versions, the OS version must
'* be WIN 2K and above. For the module alone some lower OS versions would suffice
'* too, but DirectCOM.dll uses other apis that depend on OS versions >= 2K.
'*****

Private Declare Function GetModuleHandle Lib "kernel32.dll" Alias "GetModuleHandleW" _
                  (ByVal DllName As Long) As Long

Private Declare Function LoadLibrary Lib "kernel32.dll" Alias "LoadLibraryW" _
                  (ByVal LibFilePath As Long) As Long

Private Declare Function FreeLibrary Lib "kernel32.dll" _
                  (ByVal hLibModule As Long) As Long

Private Declare Function PathCombine Lib "shlwapi.dll" Alias "PathCombineW" _
                  (ByRef CombinedPath As Byte, _
                   ByVal BaseDirPath As Long, _
                   ByVal RelFilePath As Long) As Long

Private Declare Function PathIsDirectory Lib "shlwapi.dll" Alias "PathIsDirectoryW" _
                  (ByVal Path As Long) As Long

'Suitable for absolute pathes on NTFS and non NTFS (eg FAT and FAT32) file systems,
'also suitable for absolute UNC pathes

Private Const MAX_ABSOLUTE_PATH = 2 ^ 15 - 1 'NTFS: 32767 FAT/FAT32: 260

'Relative pathes still have to be smaller, no matter what file system used!
'Note that the path length is including the trailing null character!
'So the pathes string len must be < 260 characters!

Private Const MAX_RELATIVE_PATH = 260

'Following const is needed to be able to *use* the longer pathes
'See eg. http://msdn.microsoft.com/en-us/library/aa365247.aspx: "File Names, Paths, and Namespaces"
'Note that this prefix can be used with absolute pathes only!

'Local pathes: eg. "D:\"<path><NULLCHAR>             -> "\\?\D:\"<path><NULLCHAR>
'UNC   pathes: eg. "\\server\share\"<path><NULLCHAR> -> "\\?\UNC\server\share\"<path><NULLCHAR>

Private Const NTFS_PREFIX As String = "\\?\"


'*******************************************************************************
'* Private consts
'*******************************************************************************

Private Const mcMODULENAME          As String = "modRC3"
Private Const mcERRBASENUM          As Long = vbObjectError + &H100


'*******************************************************************************
'* Public consts (made public to pleasure you :-))
'*******************************************************************************

'The names of the three framework dlls. Adapt them if dll names change

Public Const gcRC3_DirectCOM_DLLNAME   As String = "DirectCOM.dll"
Public Const gcRC3_RichClient3_DLLNAME As String = "dhRichClient3.dll"
Public Const gcRC3_SQLite_DLLNAME      As String = "sqlite36_engine.dll"


'*******************************************************************************
'* Private membervars
'*******************************************************************************

Private mFactory As dhRichClient3.cFactory

Private mIsInitialized As Boolean
Private mUsesRegistered As Boolean
Private mFrameworkPath As String


'*******************************************************************************
'* Public membervars
'*******************************************************************************

'After successful initialization the below two members hold references
'to the dhRichClient3 class instances.

'Use the functions of cConstructor to get new instances of the classes
'available in the dhRichClient3 dll.
'Most of them act like a constructor (hence the name): parameters can be passed

Public gRC3_New As dhRichClient3.cConstructor

'Use the GetInstance and GetInstanceEx functions of cRegfree for regfree
'instantiation. They are wrappers of the corresponding functions in DirectCOM.

Public gRC3_RegFree As dhRichClient3.cRegFree


'*******************************************************************************
'* Public properties
'*******************************************************************************

'Not really necessary. But its there :-)

Public Property Get gRC3_Factory() As dhRichClient3.cFactory
Set gRC3_Factory = mFactory
End Property

'One can see if the module has been initialized.

Public Property Get gRC3_IsInitialized() As Boolean
gRC3_IsInitialized = mIsInitialized
End Property

'Shows if the registered version of dhRichClient3.dll is used.

Public Property Get gRC3_UsesRegistered() As Boolean
gRC3_UsesRegistered = mUsesRegistered
End Property

'Shows the current value of the FrameworkPath. It does not show
'wether this has been used or wether it matters!

Public Property Get gRC3_FrameworkPath() As String
gRC3_FrameworkPath = mFrameworkPath
End Property



'*******************************************************************************
'* Public module initialization and deinitialization methods
'*******************************************************************************

'Call this once in Sub Main or eg Form_Load. In any case it must have been called
'before the dhRichClient3 framework can be used.

'The workings of gRC3_Init depend on wether the helper methods of the framework dll
'dhRichClient3 are instantiated using the common way (over the registry) or in the
'regfree way (not using the registry). The result however is in practical all
'aspects the same.

'*****
'* When using the registry
'*****

'- dhRichClient3.dll must be registered
'- the value of FrameworkPath does not matter, it is ignored

'An instance of cFactory is made from the registered dhRichClient3 and stored in mFactory.
'From mFactory instances of cConstructor and cRegFree are retrieved and stored in
'gRC3_New and gRC3_RegFree. Done.

'*****
'* When using the regfree way
'*****

'- dhRichClient3.dll may be registered or not, it doesn't matter
'- FrameworkPath must point to the folder that contains the three framework dlls

'FrameworkPath may be an absolute path or a relative path (relative to App.Path).

'A relative path starts with a backslash "\". It then is combined with App.Path
'to yield an absolute path.

'An absolute path either starts with "\\" (then it's an UNC path) or not with "\".
'In both cases the path is taken as is.

'If FrameworkPath is ommitted or empty, its value is set to App.Path.

'After from FrameworkPath the absolute path is build or taken, it is checked:

'- is it pointing to an existing and accessible folder?
'- are the framework dlls in this folder?

'When the checks fail, an error is raised.

'Then mFactory, gRC3_New and gRC3_RegFree are populated as above by using
'the copy of dhRichClient3.dll located in the framework folder.

'*****
'* Parameters
'*****

'FrameworkPath: see above

'ForceRegFree: if True, the regfree way is used
'              if False, it depends in which context the application is running:
'              - the regfree way is used when running compiled outside the IDE
'              - the registry is used when running inside the IDE

Public Sub gRC3_Init(Optional ByVal FrameworkPath As String, _
                     Optional ByVal ForceRegFree As Boolean)
Dim DirectCOMErrMsg As String, Path As String

On Error GoTo MethodError

If mIsInitialized Then Exit Sub

If ForceRegFree Then mUsesRegistered = False Else mUsesRegistered = gRC3_InIde

If mUsesRegistered Then 'Get cFactory from registered dhRichClient3.dll
  Set mFactory = New dhRichClient3.cFactory
Else 'Get cFactory reg free from FrameworkPath\dhRichClient3.dll
  If MakePath(FrameworkPath) Then Path = FrameworkPath Else Path = NTFS_PREFIX & FrameworkPath
  CheckPath Path
  LoadDlls Path, gcRC3_DirectCOM_DLLNAME
  Set mFactory = GetInstanceEx(StrPtr(FrameworkPath & gcRC3_RichClient3_DLLNAME), StrPtr("cFactory"))
  DirectCOMErrMsg = GETINSTANCELASTERROR
  If Len(DirectCOMErrMsg) > 0 Then ErrRaise "Error when calling GetInstanceEx:" & vbCrLf & DirectCOMErrMsg
End If

If mFactory Is Nothing Then ErrRaise "Could not instantiate cFactory"
Set gRC3_New = mFactory.C: Set gRC3_RegFree = mFactory.regfree
mFrameworkPath = FrameworkPath: mIsInitialized = True

Exit Sub

MethodError:
  Destroy
  ErrReRaise
End Sub

'Normally it is not neccessary to call gRC3_DeInit. However you can call it if
'you want to save some memory. When called after gRC3_Init has been run successfully,
'afterwards all framework dlls should have been unloaded (such freeing memory).

'!!Warning!!

'When the framework dll has been loaded reg free, do not call DeInit until all object
'references from dhRichClient3.dll your application is holding are set to Nothing too!
'If outside this module any reference to an dhRichClient3.dll object still exists when
'calling gRC3_DeInit, your application most likely will have a really nasty crash.

Public Sub gRC3_DeInit()
If mIsInitialized Then
  Destroy
  mIsInitialized = False
End If
End Sub


'*******************************************************************************
'* Public helpers that may be used independently of the core modules functionality
'*******************************************************************************

Public Function gRC3_InIde() As Boolean
On Error Resume Next
Debug.Print 1 / 0
gRC3_InIde = Err.Number
Err.Clear
End Function


'*******************************************************************************
'* Private helpers
'*******************************************************************************

Private Sub Destroy()
Set mFactory = Nothing
Set gRC3_New = Nothing
Set gRC3_RegFree = Nothing
If Not mUsesRegistered Then
  UnloadDlls gcRC3_RichClient3_DLLNAME, gcRC3_SQLite_DLLNAME, gcRC3_DirectCOM_DLLNAME
End If
End Sub

'Makes an absolute path from the given Path parameter.

'Precautions:

'- App.Path must not be an UNC path, but an absolute path to the local filesystem!
'- If Path is an UNC path, it must be an absolute path!

'Resulting value of Path (after the call to MakePath):

'- If Path was empty, it is replaced with App.Path.
'- If Path started with a "\", it is interpreted as relative path to App.Path, and
'  both pathes are combined to an absolute path.
'- If Path started with "\\" it is interpreted as UNC path and taken as is.
'- In any other case the Path is taken as is.

'Raises errors if anything goes wrong.

' ""       -> App.Path
' "\\..."  -> UNC path taken as is
' "\..."   -> is interpreted as path relative to App.Path. The leading "\" is removed and the rest
'             is combined with App.Path

'Results for example valid relative pathes, with App.Path = "E:\Folder\SubFolder\AppFolder":

' "\"                                  -> "E:\Folder\SubFolder\AppFolder\"
' "\Bin"                               -> "E:\Folder\SubFolder\AppFolder\Bin\"
' "\Bin\"                              -> "E:\Folder\SubFolder\AppFolder\Bin\"
' "\.."                                -> "E:\Folder\SubFolder\"
' "\..\"                               -> "E:\Folder\SubFolder\"
' "\..\Common"                         -> "E:\Folder\SubFolder\Common\"
' "\..\Common\"                        -> "E:\Folder\SubFolder\Common\"
' "\..\Common\Bin"                     -> "E:\Folder\SubFolder\Common\Bin\"
' "\..\Common\Bin\"                    -> "E:\Folder\SubFolder\Common\Bin\"
' "\..\.."                             -> "E:\Folder\"
' "\..\..\"                            -> "E:\Folder\"
' "\..\..\Other"                       -> "E:\Folder\Other\"

'Relative pathes must not start with a drive letter followed by a colon!

' "\C"                                 -> "E:\Folder\SubFolder\AppFolder\C\"
' "\C:"                                -> "C:\" !!!!
' "\C:\"                               -> "C:\" !!!!
' "\C:\Windows"                        -> "C:\Windows\" !!!!

'Returns True if the resulting path is an UNC path

Private Function MakePath(ByRef Path As String) As Boolean
Dim Buf() As Byte

Path = Trim$(Path)

If Len(Path) = 0 Then
  Path = App.Path
ElseIf Left$(Path, 1) = "\" Then
  If Mid$(Path, 2, 1) = "\" Then 'UNC path
    MakePath = True
  Else 'Relative path to the local filesystem
    If Len(Path) >= MAX_RELATIVE_PATH Then ErrRaise "Invalid relative path (path too long)"
    ReDim Buf(0 To MAX_ABSOLUTE_PATH * 2 - 1)
    If PathCombine(Buf(0), StrPtr(App.Path), StrPtr(Mid$(Path, 2))) = 0 Then _
       ErrRaise "Invalid relative path"
    Path = Buf
    Path = Left$(Path, InStr(Path, vbNullChar) - 1)
  End If
End If

If Right$(Path, 1) <> "\" Then Path = Path & "\"

End Function

'Checks wether the absolute Path is a valid and accessible directory

Private Sub CheckPath(ByRef Path As String)
If PathIsDirectory(StrPtr(Path)) = 0 Then _
   ErrRaise "Not a directory or directory is currently inaccessible"
CheckDlls Path, gcRC3_RichClient3_DLLNAME, gcRC3_SQLite_DLLNAME, gcRC3_DirectCOM_DLLNAME
End Sub

'Path points to a valid directory. Checks wether in this directory the framework dlls are in.

'Using Dir$ probably is flawed as it likely translates UNICODE to ANSII.
'PathFileExists (shlwapi) would be better, however it does not work with UNC-pathes:

' It works only on the local file system or on a remote drive that has been mounted to
' a drive letter. It will return FALSE for remote file paths that begin with the UNC names
' \\server or \\server\share. It will also return FALSE if a mounted remote drive is out
' of service.

'The last resort would be to actually trying to open the files using the file apis.

Private Sub CheckDlls(ByRef Path As String, ParamArray DllNames() As Variant)
Dim DllName As Variant
For Each DllName In DllNames
  If Len(Dir$(Path & DllName)) = 0 Then ErrRaise DllName & " not found in " & Path
Next DllName
Dir$ "nul" 'Release opened folder
End Sub

Private Sub LoadDlls(ByRef Path As String, ParamArray DllNames() As Variant)
Dim Res As Long, DllName As String, i As Long
For i = 0 To UBound(DllNames)
  DllName = DllNames(i)
  Res = GetModuleHandle(StrPtr(DllName))
  If Res = 0 Then Res = LoadLibrary(StrPtr(Path & DllName))
  If Res = 0 Then ErrRaise "Could not load library " & DllName
Next i
End Sub

Private Sub UnloadDlls(ParamArray DllNames() As Variant)
Dim Res As Long, DllName As String, i As Long
For i = 0 To UBound(DllNames)
  DllName = DllNames(i)
  Res = GetModuleHandle(StrPtr(DllName))
  While Res <> 0
    FreeLibrary Res
    Res = GetModuleHandle(StrPtr(DllName))
  Wend
Next i
End Sub

Private Sub ErrReRaise(Optional ByVal MethodName As String = "gRC3_Init")
If Err.Number <> mcERRBASENUM Then
  Err.Description = "Unexpected error" & vbCrLf & Err.Description
  Err.Source = mcMODULENAME & "." & MethodName & vbCrLf & Err.Source
End If
Err.Raise Err.Number, Err.Source, Err.Description
End Sub

Private Sub ErrRaise(ByVal Msg As String, Optional ByVal MethodName As String = "gRC3_Init")
Err.Raise mcERRBASENUM, mcMODULENAME & "." & MethodName, Msg
End Sub
